<?php
/**
 * Midtrans Payment Callback File
 *
 * Call back used to retrieve transaction HTTP notification, then check validity using 
 * GetStatus API Call
 * validating an Invoice ID, checking for the existence of a Transaction ID,
 * Logging the Transaction for debugging and Adding Payment to an Invoice.
 *
 * For more information, please refer to the online documentation.
 * @see http://docs.midtrans.com
 *
 * Module developed based on official WHMCS Sample Payment Gateway Module
 * 
 * @author rizda.prasetya@midtrans.com
 */


// Require libraries needed for gateway module functions.
require_once __DIR__ . '/../../../init.php';
require_once __DIR__ . '/../../../includes/gatewayfunctions.php';
require_once __DIR__ . '/../../../includes/invoicefunctions.php';

// Require Veritrans Library
require_once __DIR__ . '/../veritrans-lib/Veritrans.php';

// Detect module name from filename.
$gatewayModuleName = basename(__FILE__, '.php');

// Fetch gateway configuration parameters.
$gatewayParams = getGatewayVariables($gatewayModuleName);

// Die if module is not active.
if (!$gatewayParams['type']) {
    die("Module Not Activated");
}

/**
 * Validate callback authenticity.
 */

// Create veritrans notif object from HTTP POST notif
Veritrans_Config::$isProduction = ($gatewayParams['environment'] == 'on') ? true : false;
Veritrans_Config::$serverKey = $gatewayParams['serverkey'];
$veritrans_notification = new Veritrans_Notification();

$transaction_status = $veritrans_notification->transaction_status;
$order_id = $veritrans_notification->order_id;
$invoiceId = str_replace('payment-', '', $order_id);
$payment_type = $veritrans_notification->payment_type;
$paymentAmount = $veritrans_notification->gross_amount;
$transactionId = $veritrans_notification->transaction_id;
$paymentFee = 0;

/**
 * Validate Callback Invoice ID.
 *
 * Checks invoice ID is a valid invoice number. Note it will count an
 * invoice in any status as valid.
 *
 * Performs a die upon encountering an invalid Invoice ID.
 *
 * Returns a normalised invoice ID.
 */
$invoiceId = checkCbInvoiceID($invoiceId, $gatewayParams['name']);

/**
 * If useInvoiceAmountAsPaid enabled, use the invoice amount, as paid amount
 * this to avoid currency conversion issue on non IDR WHMCS transaction
 */

if ($gatewayParams['useInvoiceAmountAsPaid'] == 'on') {
  $invoice_result = mysql_fetch_assoc(select_query('tblinvoices', 'total, userid', array("id"=>$order_id)));
  $invoice_amount = $invoice_result['total'];
  $paymentAmount = $invoice_amount;
}

/**
 * Of tryToConvertCurrencyBack enabled
 * Try to convert amount back to IDR, if not IDR
 */
if ($gatewayParams['tryToConvertCurrencyBack'] == 'on') {
  try {
    // Fetch invoice to get the amount and userid
    $invoice_result = mysql_fetch_assoc(select_query('tblinvoices', 'total, userid', array("id"=>$order_id))); 
    $invoice_amount = $invoice_result['total'];
    // Check if amount is IDR, convert if not.
    $client_result = mysql_fetch_assoc(select_query('tblclients', 'currency', array("id"=>$invoice_result['userid'])));
    $currency_id = $client_result['currency'];
    $idr_currency_id = $gatewayParams['convertto'];
    if($currency_id != $idr_currency_id) {
        $converted_amount = convertCurrency(
          $veritrans_notification->gross_amount, 
          $idr_currency_id, 
          $currency_id
        );
    } else {
        $converted_amount = $veritrans_notification->gross_amount;
    }
    $paymentAmount = $converted_amount;
  } catch (Exception $e) {
    echo "fail to tryToConvertCurrencyBack";
  }
}

/**
 * Log Transaction.
 *
 * Add an entry to the Gateway Log for debugging purposes.
 *
 * The debug data can be a string or an array. In the case of an
 * array it will be
 *
 * @param string $gatewayName        Display label
 * @param string|array $debugData    Data to log
 * @param string $transactionStatus  Status
 */
logTransaction($gatewayParams['name'], print_r($veritrans_notification,true), $transaction_status);


// trx status handler
$success = false;
if ($veritrans_notification->transaction_status == 'capture') {
  if ($veritrans_notification->fraud_status == 'accept') {
    checkCbTransID($transactionId);
    $success = true;
  }
  else if ($veritrans_notification->fraud_status == 'challenge') {
    $success = false;
  }
}
else if ($veritrans_notification->transaction_status == 'cancel') {
  $success = false;
}
else if ($veritrans_notification->transaction_status == 'expire') {
    if ($veritrans_notification->payment_type == 'credit_card') {
        $success = false;
    }
}
else if ($veritrans_notification->transaction_status == 'deny') {
  $success = false;
}
else if ($veritrans_notification->transaction_status == 'settlement') {
  if($veritrans_notification->payment_type == 'credit_card'){
    die("Credit Card Settlement Notification Received");
  }
  else{
    checkCbTransID($transactionId);
    $success = true;
  }
}
else if ($veritrans_notification->transaction_status == 'pending') {
  $success = false;
}

function sendDiscordWebhook($transaction)
{
    $webhookUrl = 'https://discord.com/api/webhooks/1320980667424571434/1QJmkQkG5XQY9LERcsuvQzN8O56Zcux7dcgRrcD0oIpJWh3OblmV7fio0kFchfs7iQc9';
    
    $payload = [
        'content' => null,
        'embeds' => [
            [
                'title' => "**#Invoice Log**",
                'description' => 'Receiving Payment From:',
                'color' => 4627931,
                'fields' => [
                    [
                        'name' => '👤 USERNAME',
                        'value' => "```" . $transaction['user']['fullname'] . "```",
                        'inline' => true
                    ],
                    [
                        'name' => '👤 Email',
                        'value' => "```" . $transaction['user']['email'] . "```",
                        'inline' => true
                    ],
                    [
                        'name' => '💳 Transaction ID',
                        'value' => "```" . 'payment-'.$transaction['id'] . "```"
                    ],
                    [
                        'name' => '📦 Product',
                        'value' => "```" . $transaction['items']['item'][0]['description'] . "```",
                    ],
                    [
                        'name' => '📆 INVOICE DATE',
                        'value' => "```" . date('d/m/Y', strtotime($transaction['invoice_date'])) . "```",
                    ],
                    [
                        'name' => '💸 PRICE',
                        'value' => "```" . 'Rp'.$transaction['price'] . "```",
                    ],
                    [
                        'name' => '📊 Status',
                        'value' => "```" . $transaction['status'] . "```"
                    ],
                ],
                'timestamp' => date('Y-m-d H:i:s'),
                'image' => [
                    'url' => 'https://files.catbox.moe/cjnrvb.png'
                ]
            ]
        ],
        'attachments' => []
    ];

    $jsonPayload = json_encode($payload);

    $ch = curl_init($webhookUrl);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonPayload);

    $response = curl_exec($ch);

    if (curl_errno($ch)) {
        echo 'Error:' . curl_error($ch);
    }

    curl_close($ch);
}

function getUserDetailsById($userId)
{
    $command = 'GetClientsDetails';
    $postData = array(
        'clientid' => $userId,
    );
    $results = localAPI($command, $postData, $adminUsername);

    if ($results['result'] == 'success') {
        $fullname = $results['client']['fullname'];
        $email = $results['client']['email'];
        return [
            'fullname' => $fullname,
            'email' => $email,
        ];
    } else {
        return [
            'error' => $results['message'],
        ];
    }
}

function processPaymentAndSendWebhook($invoiceId)
{
    $command = 'GetInvoice';
    $postData = array(
        'invoiceid' => $invoiceId,
    );

    $results = localAPI($command, $postData, $adminUsername);

    if ($results['result'] == 'success') {
        $userId = $results['userid'];

        $userDetails = getUserDetailsById($userId);

        $transaction = [
            'id' => $results['invoiceid'],
            'status' => $results['status'],
            'invoice_date' => $results['date'],
            'due_date' => $results['duedate'],
            'price' => $results['total'],
            'user' => [
                'fullname' => $userDetails['fullname'],
                'email' => $userDetails['email']
            ],
            'items' => $results['items'],
        ];

        sendDiscordWebhook($transaction);

        echo "Payment success notification accepted";
    } else {
        echo "Error: " . $results['message'];
    }
}

if ($success) {

    /**
     * Add Invoice Payment.
     *
     * Applies a payment transaction entry to the given invoice ID.
     *
     * @param int $invoiceId         Invoice ID
     * @param string $transactionId  Transaction ID
     * @param float $paymentAmount   Amount paid (defaults to full balance)
     * @param float $paymentFee      Payment fee (optional)
     * @param string $gatewayModule  Gateway module name
     */
    addInvoicePayment(
        $invoiceId,
        $transactionId,
        $paymentAmount,
        $paymentFee,
        $gatewayModuleName
    );
    processPaymentAndSendWebhook($invoiceId);
    // error_log("=====================".$invoiceId."-".$transactionId."-".$paymentAmount."-".$paymentFee."-".$gatewayModuleName); //debugan
    echo "Payment success notification accepted";

}
else{
    die("Payment failed, denied or pending");
}
